% Copyright (c) 2015, Liverpool John Moores University
% All rights reserved.
% 
% Redistribution and use in source and binary forms, with or without modification,
% are permitted provided that the following conditions are met:
% 
% 1. Redistributions of source code must retain the above copyright notice, 
% this list of conditions and the following disclaimer.
% 
% 2. Redistributions in binary form must reproduce the above copyright notice, 
% this list of conditions and the following disclaimer in the documentation 
% and/or other materials provided with the distribution.
% 
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
% AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
% IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
% ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
% LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
% CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
% GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
% HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT 
% LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY 
% OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


% ************* LJMU_IKD_analyser **********


% Version 1.0

% GUI based software for analysing isokinetic dynamometry data generated by 
% both Biodex and Cybex (HUMAC) hardware. 

% The software reads in the raw velocity, angle and torque versus time data
% from the ssved IKD text files then displays them on three separate plots. 
% The user then needs to select which areas of the data they are interested 
% in and the software plots the selected torque against angle curves and fits
% them either with a quartic polynomial or a smoothed moving average. 
%
% The user can either manually set the required velocity range and manually
% select which resulting regions they want to analyse or they can use a
% semi-autometed routine which selects data via a sequentially applying
% velocity, angle (Range of motion) and tolerance limits. Either way the
% resulting torwu-angle fits are displayd graphically and can be saved to a
% csv file. Additionally, a number of these fit files can be selected and 
% merged into a single file suitable for statisitical analysis.
%
% The software development has proceeded in a fairly organic manner as
% various requirements have been added as we go along. As a result
% the design is a little but haphazard with, at times, fairly arbitrary 
% separation of functionality.

%Global variables used are:
% 
%  baseDirectory - location of directory to save/read data to/from

%  d - raw data storage matrix, should really call it something better!!
%  dType - string identifier of type of data i.e. 'eccentric' or concentric'
%  dProcessed - composite array of selected time, torque, angle, velocity, 
%               time and index values used during processing of data 
%  dFormat - string identifier for type of input data: biodex, cybex or other
%  dToSave - contains the selected raw data to save to file or to fit to

%  TIMECOL - Constant containing identifier of time column in raw data, d
%  TORQUECOL  - Constant containing identifier of torque column in raw data, d
%  ANGLECOL - Constant containing identifier of angle column in raw data, d
%  VELCOL - Constant containing identifier of velocity column in raw data, d
%  INDEXCOL - Constant containing identifier of index column in raw data, d

%  velocityLimits - user chosen velocity limits to limit data processing range
%  numSections - number of contiguous sections of data to process
%  sectionPoints - contains start and end indexes for all sections
%  useSection - contains identifiers of which sections to process
%  minTorqueValues - contains minimum torque values for each section
%  maxTorqueValues - contains maximum torque values for each section
%  maxTimeValues- contains maximum time values for each section
%  minTimeValues - contains minimum time values for each section
%  maxVelocityValues - contains maximum velocity values for each section
%  minVelocityValues - contains minimum velocity values for each section
%  maxAngleValues - contains maximum angle values for each section
%  minAngleValues - contains minimum angle values for each section

%  smoothVal  - number of data points to average over



 
function varargout = LJMU_IKD_analyserGUI(varargin)
% LJMU_IKD_ANALYSERGUI M-file for LJMU_IKD_analyserGUI.fig
%      LJMU_IKD_ANALYSERGUI, by itself, creates a new LJMU_IKD_ANALYSERGUI or raises the existing
%      singleton*.
%
%      H = LJMU_IKD_ANALYSERGUI returns the handle to a new LJMU_IKD_ANALYSERGUI or the handle to
%      the existing singleton*.
%
%      LJMU_IKD_ANALYSERGUI('CALLBACK',hObject,eventData,handles,...) calls the local
%      function named CALLBACK in LJMU_IKD_ANALYSERGUI.M with the given input arguments.
%
%      LJMU_IKD_ANALYSERGUI('Property','Value',...) creates a new LJMU_IKD_ANALYSERGUI or raises the
%      existing singleton*.  Starting from the left, property value pairs are
%      applied to the GUI before LJMU_IKD_analyserGUI_OpeningFcn gets called.  An
%      unrecognized property name or invalid value makes property application
%      stop.  All inputs are passed to LJMU_IKD_analyserGUI_OpeningFcn via varargin.
%
%      *See GUI Options on GUIDE's Tools menu.  Choose "GUI allows only one
%      instance to run (singleton)".
%
% See also: GUIDE, GUIDATA, GUIHANDLES

% Edit the above text to modify the response to help LJMU_IKD_analyserGUI

% Last Modified by GUIDE v2.5 15-Oct-2015 12:01:34

% Begin initialization code - DO NOT EDIT
gui_Singleton = 1;
gui_State = struct('gui_Name',       mfilename, ...
                   'gui_Singleton',  gui_Singleton, ...
                   'gui_OpeningFcn', @LJMU_IKD_analyserGUI_OpeningFcn, ...
                   'gui_OutputFcn',  @LJMU_IKD_analyserGUI_OutputFcn, ...
                   'gui_LayoutFcn',  [] , ...
                   'gui_Callback',   []);
if nargin && ischar(varargin{1})
    gui_State.gui_Callback = str2func(varargin{1});
end

if nargout
    [varargout{1:nargout}] = gui_mainfcn(gui_State, varargin{:});
else
    gui_mainfcn(gui_State, varargin{:});
end
% End initialization code - DO NOT EDIT


% --- Executes just before LJMU_IKD_analyserGUI is made visible.
function LJMU_IKD_analyserGUI_OpeningFcn(hObject, eventdata, handles, varargin)
% This function has no output args, see OutputFcn.
% hObject    handle to figure
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)
% varargin   command line arguments to LJMU_IKD_analyserGUI (see VARARGIN)

% Choose default command line output for LJMU_IKD_analyserGUI
handles.output = hObject;

% Update handles structure
guidata(hObject, handles);
global baseDirectory;

% baseDirectory = 'C:\Documents and Settings\Administrator\My Documents\MATLAB\Dual Input\db\';
 baseDirectory = 'M:\My Documents\Matlab\IKD Analyser\';

 set(findobj('Tag','filenameDirEdit'),'String', baseDirectory);
 
% UIWAIT makes LJMU_IKD_analyserGUI wait for user response (see UIRESUME)
% uiwait(handles.figure1);


% --- Outputs from this function are returned to the command line.
function varargout = LJMU_IKD_analyserGUI_OutputFcn(hObject, eventdata, handles) 
% varargout  cell array for returning output args (see VARARGOUT);
% hObject    handle to figure
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Get default command line output from handles structure
varargout{1} = handles.output;


    % --- Executes on button press in loadButton.
function loadButton_Callback(hObject, eventdata, handles)
% hObject    handle to loadButton (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Load raw IKD data in form of txt file and save in 'd' matrix

global baseDirectory;
global d;
global dType
global dProcessed;
global TIMECOL;
global TORQUECOL;
global ANGLECOL;
global VELCOL;
global INDEXCOL;
global dFormat;

%Define column 'static' variables
TIMECOL = 1; TORQUECOL =2; ANGLECOL = 3; VELCOL = 4; INDEXCOL =5;

%Get file name
set(findobj('Tag','loadButton'),'Enable', 'off');
drawnow;
[FileName,PathName] = uigetfile({'*.txt;*.csv;*.xls;*.xlsx', 'Data Files (*.txt;*.csv;*.xls;*.xlsx)'; ...
     }, 'Select the file to display', baseDirectory);
 if FileName == 0
    set(findobj('Tag','loadButton'),'Enable', 'on');
    drawnow;
    return;
 else
    baseDirectory = PathName;
    set(findobj('Tag','filenameEdit'),'String', FileName);
    set(findobj('Tag','buttonApply'),'Enable', 'on');
    set(findobj('Tag','buttonManualSelect'),'Enable', 'off');
    set(findobj('Tag','buttonSelectAll'),'Enable', 'off');
    set(findobj('Tag','buttonDisplay'),'Enable', 'off');
    set(findobj('Tag','buttonSave'),'Enable', 'off');
    set(findobj('Tag','buttonCurveFit'),'Enable', 'off');
    set(findobj('Tag','buttonSaveFit'),'Enable', 'off');
 end
 
 %Ask user to specify type of file
  button =  questdlg('Select data type','','Concentric','Eccentric', 'Concentric');
  if strcmp(button,'')  
      set(findobj('Tag','loadButton'),'Enable', 'on');
      drawnow;
      return
  end
 %Display file type and set default values for fitting   
 if strcmp(button,'Eccentric') 
     dType = 'Eccentric';
     set(findobj('Tag','radioEccentric'),'Value', 1);
     set(findobj('Tag','editVelTol'),'String', '10');
     set(findobj('Tag','editROM'),'String', '65');
     set(findobj('Tag','editROMOmitFirst'),'String', '0');
     set(findobj('Tag','editROMOmitLaat'),'String', '0');
     set(findobj('Tag','editROMTol'),'String', '10');
     set(findobj('Tag','editTorTol'),'String', '10');
 else
     dType = 'Concentric';
     set(findobj('Tag','radioConcentric'),'Value', 1);
     set(findobj('Tag','editVelTol'),'String', '10');
     set(findobj('Tag','editROM'),'String', '80');
     set(findobj('Tag','editROMTol'),'String', '10');
     set(findobj('Tag','editTorTol'),'String', '10');
     set(findobj('Tag','editROMOmitFirst'),'String', '0');
     set(findobj('Tag','editROMOmitLaat'),'String', '0');
 end
 set(findobj('Tag','editNumPosRegions'),'String', '0');
 set(findobj('Tag','editNumNegRegions'),'String', '0');
 
%Read in file data and display            
if FileName ~= 0
    if get(findobj('Tag','radioBiodex'),'Value')== 1
        %Define column 'static' variables
        TIMECOL = 1; TORQUECOL =2; ANGLECOL = 3; VELCOL = 4; INDEXCOL =5;
        dProcessed = []; d=[];
        d = dlmread(strcat(PathName, FileName), '' , 6,0);
        %Delete unused column
        d(:,3) = [];
        dFormat = 'Biodex';
    elseif get(findobj('Tag','radioCybex'),'Value')== 1
        %Define column 'static' variables
        TIMECOL = 1; TORQUECOL =3; ANGLECOL = 2; VELCOL = 4; INDEXCOL =5;
        [d , txt, raw] = xlsread(strcat(PathName, FileName));
        %Convert to msec
        d(:,1)=[];
        d(:,1) = d(:,1)*1000;
        dFormat = 'Cybex';
        %Converts +ve torques to -ve when angle is increasing
        for i = 1:length(d(:,1))-1
            if d(i,ANGLECOL) < d(i+1,ANGLECOL)
                d(i,TORQUECOL) = -d(i,TORQUECOL);
            end       
        end
    elseif get(findobj('Tag','radioOther'),'Value')== 1
        %Define column 'static' variables
        TIMECOL = 1; TORQUECOL =2; ANGLECOL = 3; VELCOL = 4; INDEXCOL =5;
        d = dlmread(strcat(PathName, FileName), '' , 2,0);
        %Convert to msec
        d(:,1) = d(:,1)*10;
        dFormat = 'Other';
    end
    displayRawData;
end
set(findobj('Tag','loadButton'),'Enable', 'on');
set(findobj('Tag','buttonAutoSelect'), 'Enable', 'On');
set(findobj('Tag','buttonAutoFit'), 'Enable', 'Off');

%Find peaks of velocity data and display
temp=[];
for i = 1:length(d)
    if abs(d(i,VELCOL)) ~= 0
        temp= [temp d(i,VELCOL)];
    end
end
velMax=round(mode(abs(temp))/10)*10;
velMin=-velMax;
set(findobj('Tag', 'editMax'), 'String', num2str( velMax));
set(findobj('Tag', 'editMin'), 'String', num2str( velMin));
set(findobj('Tag', 'editMaxVelocity'), 'String', num2str( velMax));
set(findobj('Tag', 'editMinVelocity'), 'String', num2str( velMin));
drawnow;


function displayRawData

% Displays raw data as Torque, Angle and Velocity against Time

global d;
global TIMECOL;
global TORQUECOL;
global ANGLECOL;
global VELCOL;

subplot(3,1,1), plot(d(:, TIMECOL),d(:, TORQUECOL) );
xlabel('Time (mSec.)');    ylabel('Torque (N m)');
subplot(3,1,2), plot(d(:, TIMECOL),d(:, ANGLECOL) );
xlabel('Time (mSec.)');    ylabel('Angle (Degrees)');
subplot(3,1,3), plot(d(:, TIMECOL),d(:, VELCOL) );  
xlabel('Time (mSec.)');    ylabel('Velocity. (Deg/sec)');
drawnow
grid on;




function filenameDirEdit_Callback(hObject, eventdata, handles)
% hObject    handle to filenameDirEdit (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of filenameDirEdit as text
%        str2double(get(hObject,'String')) returns contents of filenameDirEdit as a double


% --- Executes during object creation, after setting all properties.
function filenameDirEdit_CreateFcn(hObject, eventdata, handles)
% hObject    handle to filenameDirEdit (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end



% --- Executes during object creation, after setting all properties.
function loadButton_CreateFcn(hObject, eventdata, handles)
% hObject    handle to loadButton (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called


% --- Executes on button press in changeButton.
function changeButton_Callback(hObject, eventdata, handles)
% hObject    handle to changeButton (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Allows user to change and store base directory

global baseDirectory;
baseDirectory = strcat(uigetdir( baseDirectory,'Select directory to open data from'),'\');

if baseDirectory ~= 0 
    set(findobj('Tag','filenameDirEdit'),'String', baseDirectory);
end



function editMax_Callback(hObject, eventdata, handles)
% hObject    handle to editMax (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of editMax as text
%        str2double(get(hObject,'String')) returns contents of editMax as a double


function applyVelocityFiltering(min, max, tol)

%Filters raw data, d, according to input min, max and tolerance velocity values.
%Stores selected data in dProcessed

global d; 
global dProcessed;
global velocityLimits;
global TIMECOL;
global TORQUECOL;
global ANGLECOL;
global VELCOL;
global dFormat;

%Apply user set limits to velocity data

velocityLimits = (4);
velocityLimits(1) = max  + abs(max*tol);
velocityLimits(2) = max - abs(max*tol);
velocityLimits(3) = min + abs(min*tol);
velocityLimits(4) = min - abs(min*tol);
x=[];
yVel=[];
yAng=[];
yTor=[];

%Get index of valid points into data array d
dIndex = [];
for i = 1:length(d)
    if  (d(i, VELCOL) <= velocityLimits(1) &&  d(i, VELCOL) >= velocityLimits(2))
        dIndex = [dIndex, i];
    elseif ((d(i, VELCOL) <= velocityLimits(3) &&  d(i, VELCOL) >= velocityLimits(4)))
        dIndex = [dIndex, i];
    end;
end; 

%Create subset of required data
x = d(dIndex, TIMECOL);

%Correct for gravity if required
if get(findobj('Tag','checkGravity'),'Value') == 0
    yTor= d(dIndex, TORQUECOL);
else
    g = str2num(get(findobj('Tag','editWeight'),'string'));
    angle = str2num(get(findobj('Tag','editAngle'),'string'));
    for i = 1: length(dIndex)
        Tg =(g/cos(angle))/d(dIndex(i),4);
        if d(dIndex(i), TORQUECOL) >= 0
           %Extension
           yTor = [yTor; d(dIndex(i), TORQUECOL) + Tg];
        else
           %Flexion
           yTor = [ yTor; d(dIndex(i), TORQUECOL) - Tg];
        end
    end
end

%Store in processed data array
yAng = d(dIndex, ANGLECOL);
yVel = d(dIndex, VELCOL);
if strcmp(dFormat,'Biodex')
    dProcessed = [ x, yTor, yAng, yVel, dIndex'];
elseif strcmp(dFormat,'Cybex')
    dProcessed = [ x, yAng, yTor, yVel, dIndex'];
elseif strcmp(dFormat,'Other')
    dProcessed = [ x,  yTor, yAng, yVel, dIndex'];
end


function createSectionData

%Extracts section limit indexes from processed data array (dProccessed) and stores
%them in sectionPoints global variable.

global numSections;
global sectionPoints;
global INDEXCOL;
global dProcessed;
global VELCOL;

%Create boundary data 
numSections = 1;
numPosSections = 0;
numNegSections = 0;
sectionPoints = [1];
sectionStartIndex = 1;
%Look for non-contiguous values in the index column
for i = 1: length(dProcessed)-1
   while (dProcessed(i+1,INDEXCOL ) - dProcessed(i,INDEXCOL )) ~= 1 
        numSections = numSections + 1;
        sectionPoints = [sectionPoints i (i+1)];
        if mean(dProcessed( sectionStartIndex:i, VELCOL)) > 0
            numPosSections = numPosSections + 1;
        else
            numNegSections = numNegSections + 1;
        end
        sectionStartIndex=i+1;
        break;
   end
end
sectionPoints = [sectionPoints length(dProcessed)];
if mean(dProcessed( sectionStartIndex:length(dProcessed), VELCOL)) > 0
    numPosSections = numPosSections + 1;
else
    numNegSections = numNegSections + 1;
end
% Displau number of negative and positive sections
set(findobj('Tag','editNumPosRegions'),'String', num2str(numPosSections));
set(findobj('Tag','editNumNegRegions'),'String', num2str(numNegSections));

function displaySelectedData

%Displays selected data as cyan circles on top of raw data

global dProcessed;
global d;
global TIMECOL;
global TORQUECOL;
global ANGLECOL;
global VELCOL;

%Display selected data
subplot(3,1,1), cla, hold on, plot(dProcessed(:, TIMECOL),dProcessed(:, TORQUECOL),'co' ), plot(d(:,TIMECOL), d(:,TORQUECOL)), hold off;
xlabel('Time (mSec.)');    ylabel('Torque (N m)');
subplot(3,1,2), cla, hold on, plot(dProcessed(:, TIMECOL),dProcessed(:, ANGLECOL),'co' ), plot(d(:,TIMECOL), d(:,ANGLECOL)), hold off;
xlabel('Time (mSec.)');    ylabel('Angle (Degrees)');
subplot(3,1,3), cla, hold on, plot(dProcessed(:, TIMECOL),dProcessed(:, VELCOL),'co' ), plot(d(:,TIMECOL), d(:,VELCOL)), hold off;
xlabel('Time (mSec.)');    ylabel('Velocity. (Deg/sec)');

 set(findobj('Tag','buttonManualSelect'),'Enable', 'on');
 set(findobj('Tag','buttonSelectAll'),'Enable', 'on');
 set(findobj('Tag','buttonDisplay'),'Enable', 'off');
 set(findobj('Tag','buttonSave'),'Enable', 'off');
 set(findobj('Tag','buttonCurveFit'),'Enable', 'off');
 set(findobj('Tag','buttonSaveFit'),'Enable', 'off');
 drawnow;
 
 
    
% --- Executes during object creation, after setting all properties.
function editMax_CreateFcn(hObject, eventdata, handles)
% hObject    handle to editMax (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end



function editMin_Callback(hObject, eventdata, handles)
% hObject    handle to editMin (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of editMin as text
%        str2double(get(hObject,'String')) returns contents of editMin as a double


% --- Executes during object creation, after setting all properties.
function editMin_CreateFcn(hObject, eventdata, handles)
% hObject    handle to editMin (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end



function editTol_Callback(hObject, eventdata, handles)
% hObject    handle to editTol (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of editTol as text
%        str2double(get(hObject,'String')) returns contents of editTol as a double


% --- Executes during object creation, after setting all properties.
function editTol_CreateFcn(hObject, eventdata, handles)
% hObject    handle to editTol (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end




% --- Executes during object creation, after setting all properties.
function radioAngle_CreateFcn(hObject, eventdata, handles)
% hObject    handle to radioAngle (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called


% --------------------------------------------------------------------
function uipanel17_ButtonDownFcn(hObject, eventdata, handles)
% hObject    handle to uipanel17 (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)


% --------------------------------------------------------------------
function uipanel18_ButtonDownFcn(hObject, eventdata, handles)
% hObject    handle to uipanel18 (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

velMax = str2num(get(findobj('Tag', 'editMax'), 'String'));
velMin = str2num(get(findobj('Tag', 'editMin'), 'String'));
velTol = str2num(get(findobj('Tag', 'editTol'), 'String'))/100;
applyVelocityFiltering(velMin,velMax,velTol);

%Display
displaySelectedData;



function edit16_Callback(hObject, eventdata, handles)
% hObject    handle to edit16 (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of edit16 as text
%        str2double(get(hObject,'String')) returns contents of edit16 as a double


% --- Executes during object creation, after setting all properties.
function edit16_CreateFcn(hObject, eventdata, handles)
% hObject    handle to edit16 (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end


% --- Executes during object creation, after setting all properties.
function changeButton_CreateFcn(hObject, eventdata, handles)
% hObject    handle to changeButton (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called


% --- Executes during object creation, after setting all properties.
function radioConcentric_CreateFcn(hObject, eventdata, handles)
% hObject    handle to radioConcentric (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called


% --- Executes on button press in buttonManualSelect.
function buttonManualSelect_Callback(hObject, eventdata, handles)
% hObject    handle to buttonAutoSelect (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Gets user to select which of the peaks selected by the basic velocity
% filtering they wish to use in the subsequent analysis

global d; 
global dProcessed;
global numSections;
global sectionPoints;
global minTorqueValues;
global maxTorqueValues;
global useSection;
global maxTimeValues;
global minTimeValues;
global maxVelocityValues;
global minVelocityValues;
global maxAngleValues;
global minAngleValues;
global TIMECOL;
global TORQUECOL;
global ANGLECOL;
global VELCOL;

%Initialise variables
xkeep=[];
tkeep=[];
pkeep=[];
vkeep=[];
lenkeep =0;
useSection = zeros(numSections,1);
maxTorqueValues =[];
minTorqueValues =[];
maxTimeValues=[];
minTimeValues=[];
maxVelocityValues=[];
minVelocityValues=[];
maxAngleValues=[];
minAngleValues=[];
numPosRegions = 0;
numNegRegions = 0;
j=0;
%Step through sections and get user to select peaks to save 
for i = 1:2:numSections*2
    j = j+ 1;
    t = dProcessed(sectionPoints(i):sectionPoints(i+1),TORQUECOL);
    x = dProcessed(sectionPoints(i):sectionPoints(i+1),TIMECOL);
    p = dProcessed(sectionPoints(i):sectionPoints(i+1),ANGLECOL);
    v = dProcessed(sectionPoints(i):sectionPoints(i+1),VELCOL);
    subplot(3,1,1), hold on, plot(x,t, 'go' ), plot(d(:,TIMECOL), d(:,TORQUECOL)), hold off;
    xlabel('Time (mSec.)');    ylabel('Torque (N m)');
    subplot(3,1,2), hold on, plot(x,p, 'go' ), plot(d(:,TIMECOL), d(:,ANGLECOL)), hold off;
    xlabel('Time (mSec.)');    ylabel('Angle (Degrees)');
    subplot(3,1,3), hold on, plot(x,v, 'go' ), plot(d(:,TIMECOL), d(:,VELCOL)), hold off;
    xlabel('Time (mSec.)');    ylabel('Velocity (Degs/sec)');
    button =  questdlg('Use this section of data?','','Yes','No','Cancel', 'Yes');
    if strcmp(button,'Yes')
        %Save to appropriate global storage
        useSection(j) = 1;
        xkeep = [xkeep;  x];
        tkeep = [tkeep; t];
        pkeep = [pkeep; p];
        vkeep = [vkeep; v];
        subplot(3,1,1),cla, hold on, plot(xkeep,tkeep, 'co' ), plot(d(:,TIMECOL), d(:,TORQUECOL)), hold off;
        subplot(3,1,2),cla, hold on, plot(xkeep,pkeep, 'co' ), plot(d(:,TIMECOL), d(:,ANGLECOL)), hold off;
        subplot(3,1,3),cla, hold on, plot(xkeep,vkeep, 'co' ), plot(d(:,TIMECOL), d(:,VELCOL)), hold off;
        %Max or min?
        overallMean= mean(d(:,TORQUECOL));
        if mean(t) >= overallMean
            [val, index] = max(t); 
            maxTimeValues=[maxTimeValues;x(index)];
            maxVelocityValues=[maxVelocityValues;v(index)];
            maxAngleValues=[maxAngleValues;p(index)];
            maxTorqueValues =[maxTorqueValues; max(t)];
            numPosRegions = numPosRegions + 1;
        else
            [val, index] = min(t); 
            minTorqueValues =[minTorqueValues; min(t)];
            minTimeValues=[minTimeValues;x(index)];
            minVelocityValues=[minVelocityValues;v(index)];
            minAngleValues=[minAngleValues;p(index)];
            numNegRegions = numNegRegions + 1;
        end
        if length(t) > lenkeep
            lenkeep = length(t);
        end
    elseif strcmp(button,'No')
        subplot(3,1,1), cla, hold on, plot(xkeep,tkeep, 'co' ), plot(d(:,TIMECOL), d(:,TORQUECOL)), hold off;
        subplot(3,1,2), cla, hold on, plot(xkeep,pkeep, 'co' ), plot(d(:,TIMECOL), d(:,ANGLECOL)), hold off;
        subplot(3,1,3), cla, hold on, plot(xkeep,vkeep, 'co' ), plot(d(:,TIMECOL), d(:,VELCOL)), hold off;
    else
        subplot(3,1,1), cla, hold on, plot(dProcessed(:, TIMECOL),dProcessed(:, TORQUECOL),'ro' ), plot(d(:,TIMECOL), d(:,TORQUECOL)), hold off;
        subplot(3,1,2), cla, hold on, plot(dProcessed(:, TIMECOL),dProcessed(:, ANGLECOL),'ro' ), plot(d(:,TIMECOL), d(:,ANGLECOL)), hold off;
        subplot(3,1,3), cla, hold on, plot(dProcessed(:, TIMECOL),dProcessed(:, VELCOL),'ro' ), plot(d(:,TIMECOL), d(:,VELCOL)), hold off;
        return;
    end
end
set(findobj('Tag','editNumPosRegions'),'String', num2str(numPosRegions));
set(findobj('Tag','editNumNegRegions'),'String', num2str(numNegRegions));

%Save selected data 
saveSelectedDataToMemory(max(lenkeep), length(maxTorqueValues) + length(minTorqueValues));

 set(findobj('Tag','buttonManualSelect'),'Enable', 'on');
 set(findobj('Tag','buttonSelectAll'),'Enable', 'on');
 set(findobj('Tag','buttonDisplay'),'Enable', 'on');
 set(findobj('Tag','buttonSave'),'Enable', 'on');
 set(findobj('Tag','buttonCurveFit'),'Enable', 'on');
 set(findobj('Tag','buttonSaveFit'),'Enable', 'off');

function saveSelectedDataToMemory(maxElements, maxSections )

%Saves required data to dToSave array and related variables

global dProcessed;
global numSections;
global sectionPoints;
global dToSave;
global useSection;
global TIMECOL;
global TORQUECOL;
global ANGLECOL;
global VELCOL;

%Create suitable sized NaN array
dToSave = zeros(maxElements, 4*(maxSections));
dToSave(:,:) = NaN;
j=0;
k=1;
%Fill with selected data values
for i=1:2:numSections*2
    j=j+1;
    if useSection(j)
        time = dProcessed(sectionPoints(i):sectionPoints(i+1),TIMECOL);
        tor = dProcessed(sectionPoints(i):sectionPoints(i+1),TORQUECOL);
        pos = dProcessed(sectionPoints(i):sectionPoints(i+1),ANGLECOL);
        vel = dProcessed(sectionPoints(i):sectionPoints(i+1),VELCOL);
        for m = 1:length(time)
                dToSave(m,k)= time(m);
                dToSave(m,k+1)= tor(m);
                dToSave(m,k+2)= pos(m);
                dToSave(m,k+3)= vel(m);
        end
        k=k+4;
    end
end
set(findobj('Tag','buttonSave'),'Enable', 'on');
set(findobj('Tag','buttonDisplay'),'Enable', 'on');

function editNumPosRegions_Callback(hObject, eventdata, handles)
% hObject    handle to editNumPosRegions (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of editNumPosRegions as text
%        str2double(get(hObject,'String')) returns contents of editNumPosRegions as a double


% --- Executes during object creation, after setting all properties.
function editNumPosRegions_CreateFcn(hObject, eventdata, handles)
% hObject    handle to editNumPosRegions (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end


% --- Executes on button press in buttonApply.
function buttonApply_Callback(hObject, eventdata, handles)
% hObject    handle to buttonApply (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Applies displayed velocity selection criteria and displays reulting
% torque, angle and velocity curves

global dProcessed;
global d;
global TIMECOL;
global TORQUECOL;
global ANGLECOL;
global VELCOL;

%Read in and apply velocity selection criteria
velMax = str2num(get(findobj('Tag', 'editMax'), 'String'));
velMin = str2num(get(findobj('Tag', 'editMin'), 'String'));
velTol = str2num(get(findobj('Tag', 'editTol'), 'String'))/100;
applyVelocityFiltering(velMin,velMax,velTol);
createSectionData

%Display selected data
subplot(3,1,1), cla, hold on, plot(dProcessed(:, TIMECOL),dProcessed(:, TORQUECOL),'ro' ), plot(d(:,TIMECOL), d(:,TORQUECOL)), hold off;
xlabel('Time (mSec.)');    ylabel('Torque (N m)');
subplot(3,1,2), cla, hold on, plot(dProcessed(:, TIMECOL),dProcessed(:, ANGLECOL),'ro' ), plot(d(:,TIMECOL), d(:,ANGLECOL)), hold off;
xlabel('Time (mSec.)');    ylabel('Angle (Degrees)');
subplot(3,1,3), cla, hold on, plot(dProcessed(:, TIMECOL),dProcessed(:, VELCOL),'ro' ), plot(d(:,TIMECOL), d(:,VELCOL)), hold off;
xlabel('Time (mSec.)');    ylabel('Velocity. (Deg/sec)');

 set(findobj('Tag','buttonManualSelect'),'Enable', 'on');
 set(findobj('Tag','buttonSelectAll'),'Enable', 'on');
 set(findobj('Tag','buttonDisplay'),'Enable', 'off');
 set(findobj('Tag','buttonSave'),'Enable', 'off');
 set(findobj('Tag','buttonCurveFit'),'Enable', 'off');
 set(findobj('Tag','buttonSaveFit'),'Enable', 'off');

% --- Executes on button press in buttonSave.
function buttonSave_Callback(hObject, eventdata, handles)
% hObject    handle to buttonSave (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Saves selected torque, velocity, anngle and time values to csv file
global baseDirectory;

%Get file name to save to
[filename,PathName] = uiputfile('*.csv;', 'Save file name',baseDirectory);
if filename == 0 
    return;
else
    baseDirectory = PathName;
end
%Write header
fid = fopen(strcat(baseDirectory,filename), 'w');
writeBasicHeader(fid);
%Write data
writeData(fid);

fclose(fid);

function writeData(fid)

global dToSave;
global numSections;
global sectionPoints;
global useSection;

%Write selected data to file
fprintf(fid, 'Time, Torque, Angle, Velocity\n' );
maxLength = 0;
j=1;
k=1;
for i = 1:2:numSections*2
    if useSection(k)==1
        temp = sectionPoints(i+1) - sectionPoints(i) + 1;
        if temp > maxLength
            maxLength= temp;
        end
        j=j+1;
    end
    k=k+1;
end
for i = 1:maxLength
    for k = 1:(j-1)*4
        fprintf(fid, '%f ,' , dToSave(i,k));
    end
    fprintf(fid, '\n');
end

function writeBasicHeader(fid)

%Writes basic header information (excluding fitting parameters and data) to supplied
%file id

global minTorqueValues;
global maxTorqueValues;
global maxTimeValues;
global minTimeValues;
global maxVelocityValues;
global minVelocityValues;
global maxAngleValues;
global minAngleValues;
global dType;
global velocityLimits;

fprintf(fid, 'Measurement type, % s \n', dType);
fprintf(fid, 'Velocity limits,');
for i =1:length(velocityLimits)
    fprintf(fid, ' % .2f, ', velocityLimits(i));
end
fprintf(fid, '\n');
fprintf(fid, 'Number of regions, % 5i \n', length(maxTorqueValues) + length(minTorqueValues));
if strcmp(dType, 'Concentric')
    pos = 'extension';
    neg = 'flexion';
else
    pos = 'flexion';
    neg = 'extension';
end

fprintf(fid, 'Mean %s torque, % .2f \n',  pos,mean(maxTorqueValues));
fprintf(fid, 'Mean %s torque, % .2f \n', neg, mean(minTorqueValues));
fprintf(fid, 'Individual %s peak torque values,', pos);
for i =1:length(maxTorqueValues)
    fprintf(fid, ' % .2f, ', maxTorqueValues(i));
end
fprintf(fid, '\n');
fprintf(fid, 'Individual %s peak time values,', pos);
for i =1:length(maxTorqueValues)
    fprintf(fid, ' % .2f, ', maxTimeValues(i));
end
fprintf(fid, '\n');
fprintf(fid, 'Individual %s peak angle values,', pos);
for i =1:length(maxTorqueValues)
    fprintf(fid, ' % .2f, ', maxAngleValues(i));
end
fprintf(fid, '\n');
fprintf(fid, 'Individual %s peak velocity values,', pos);
for i =1:length(maxTorqueValues)
    fprintf(fid, ' % .2f, ', maxVelocityValues(i));
end
fprintf(fid, '\n');
fprintf(fid, 'Individual %s peak torque values,', neg);
for i =1:length(minTorqueValues)
    fprintf(fid, ' % .2f, ', minTorqueValues(i));
end
fprintf(fid, '\n');
fprintf(fid, 'Individual %s peak time values,', neg);
for i =1:length(minTorqueValues)
    fprintf(fid, ' % .2f, ', minTimeValues(i));
end
fprintf(fid, '\n');
fprintf(fid, 'Individual %s peak angle values,', neg);
for i =1:length(minTorqueValues)
    fprintf(fid, ' % .2f, ', minAngleValues(i));
end
fprintf(fid, '\n');
fprintf(fid, 'Individual %s peak velocity values,', neg);
for i =1:length(minTorqueValues)
    fprintf(fid, ' % .2f, ', minVelocityValues(i));
end
fprintf(fid, '\n');


% --- Executes on button press in buttonCurveFit.
function buttonCurveFit_Callback(hObject, eventdata, handles)
% hObject    handle to buttonCurveFit (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

%Applies both averaged and polynomial fit though only displays the
%requested one

%selectAll;
if get(handles.radioAverage,'Value') == 1 
    polyCurveFitting;
    averagedCurveFitting(handles);
else
    averagedCurveFitting(handles);
    polyCurveFitting;
end 


function averagedCurveFitting(handles)

% Performs an averaged smoothed fit to the selected max and min regions and
% displays results

global dToSave;
global dType;
global maxXAveragedFit;
global minXAveragedFit;
global maxYAveragedFit;
global minYAveragedFit;
global smoothVal;

%Read in required limits and smoothing value
smoothVal = str2num(get(handles.editSmooth, 'String'));
minPos = str2num(get(findobj('Tag','editMinAnglePos'), 'String'));
maxPos = str2num(get(findobj('Tag','editMaxAnglePos'), 'String'));
minNeg = str2num(get(findobj('Tag','editMinAngleNeg'), 'String'));
maxNeg = str2num(get(findobj('Tag','editMaxAngleNeg'), 'String'));
posValuesX=[];
posValuesY=[];
negValuesX=[];
negValuesY=[];
posDataToAverage=[];
negDataToAverage=[];
numNeg=0;
numPos=0;
colsPerSection = 4;
z = length(dToSave(1,:))/colsPerSection;
subplot(2,1,1),  cla;
subplot(2,1,2),  cla;

%Display data
 for i = 2:4:(z*colsPerSection)-2
    y = dToSave(:,i);
    %Strip out NAN's
    nanIndex=isnan(y);
    ind = length(y);
    for j = 1:ind
        if nanIndex(j) == true 
            ind = j-1;
            break
        end
    end
    %Read reduced data set
    y = dToSave(1:ind,i);
    x = dToSave(1:ind,i+1);
    %Plot curve
    if mean(y) > 0%overallMean %Positive
        subplot(2,1,1),  hold on, plot(x,y, 'go' ), hold off;
        posValuesY=[posValuesY ; y];
        posValuesX=[posValuesX; x];
        numPos = numPos + 1;
        posDataToAverage=[posDataToAverage; x, y];
    else
        subplot(2,1,2),  hold on, plot(x,y, 'go' ), hold off;
        negValuesY=[negValuesY ; y];
        negValuesX=[negValuesX; x];
        numNeg = numNeg + 1;
        negDataToAverage=[negDataToAverage; x, y];
    end
 end

%Average and smoooth curves
%Positive first..
if numPos > 0
    %Average data
    posDataToAverage=sortrows(posDataToAverage,1);
    averagedData=[];
    x=0;y=0;n=0;i=1;
    while i <= length(posDataToAverage)
        if x == 0 || posDataToAverage(i,1)== x
            y=y+posDataToAverage(i,2);
            x=posDataToAverage(i,1);
            n=n+1;
        else
            averagedData=[averagedData; x, y/n];
            x= posDataToAverage(i,1);
            y= posDataToAverage(i,2);
            n=1;
        end
        i=i+1;
    end
    averagedData=[averagedData; x, y/n];
    %Plot over required range
    for i=length(averagedData):-1:1
        if averagedData(i,1) < minPos || averagedData(i,1) > maxPos
            averagedData(i,:)=[];
        end   
    end
    subplot(2,1,1),  hold on, plot(averagedData(:,1),smooth(averagedData(:,2),smoothVal), 'kx-' ), hold off;
    xlabel('Angle (Degrees)');    ylabel('Torque (N m)');
    if strcmp(dType,'Concentric')
        type= ' extension';
    else
        type = ' flexion';
    end
    [p,i]=max(smooth(averagedData(:,2),smoothVal));
    s=sprintf('   Maximum%s torque = % .4f  Nm at %.1f degrees\n',  type, p, averagedData(i,1));
    title( strcat(num2str(smoothVal), ' point smoothed average   -', s));
    %Store averaged data
    maxXAveragedFit=averagedData(:,1);
    maxYAveragedFit=smooth(averagedData(:,2),smoothVal);
end
%Negative next...
if numNeg > 0
    negDataToAverage=sortrows(negDataToAverage,1);
    averagedData=[];
    x=0;y=0;n=0;i=1;
    while i <= length(negDataToAverage)
        if x == 0 || negDataToAverage(i,1)== x
            y=y+negDataToAverage(i,2);
            x=negDataToAverage(i,1);
            n=n+1;
        else
            averagedData=[averagedData; x, y/n];
            x= negDataToAverage(i,1);
            y= negDataToAverage(i,2);
            n=1;
        end
        i=i+1;
    end
    averagedData=[averagedData; x, y/n];
    %Plot over required range
    for i=length(averagedData):-1:1
        if averagedData(i,1) < minNeg || averagedData(i,1) > maxNeg
            averagedData(i,:)=[];
        end   
    end
    subplot(2,1,2),  hold on, plot(averagedData(:,1), smooth(averagedData(:,2),smoothVal), 'kx-' ), hold off;
    if strcmp(dType,'Concentric')
        type= ' extension';
    else
        type = ' flexion';
    end
    [p,i]=max(abs(smooth(averagedData(:,2),smoothVal)));
    s=sprintf('   Maximum%s torque = % .4f  Nm at %.1f degrees\n',  type, p, averagedData(i,1));
    title( strcat( num2str(smoothVal), ' point smoothed average   -', s));
    xlabel('Angle (Degrees)');    ylabel('Torque (N m)');
    %Store averaged data
    minXAveragedFit=averagedData(:,1);
    minYAveragedFit=smooth(averagedData(:,2),smoothVal);
end

 set(findobj('Tag','buttonManualSelect'),'Enable', 'on');
 set(findobj('Tag','buttonSelectAll'),'Enable', 'on');
 set(findobj('Tag','buttonDisplay'),'Enable', 'on');
 set(findobj('Tag','buttonSave'),'Enable', 'on');
 set(findobj('Tag','buttonCurveFit'),'Enable', 'on');
 set(findobj('Tag','buttonSaveFit'),'Enable', 'on');
 
function polyCurveFitting

% Performs an quartic polynomial  fit to the selected max and min regions and
% displays results

global dToSave;
global dType;
global maxParameters;
global minParameters;
global maxXPolyFit;
global minXPolyFit;
global maxYPolyFit;
global minYPolyFit;

minPos = str2num(get(findobj('Tag','editMinAnglePos'), 'String'));
maxPos = str2num(get(findobj('Tag','editMaxAnglePos'), 'String'));
minNeg = str2num(get(findobj('Tag','editMinAngleNeg'), 'String'));
maxNeg = str2num(get(findobj('Tag','editMaxAngleNeg'), 'String'));
posValuesX=[];
posValuesY=[];
negValuesX=[];
negValuesY=[];
numNeg=0;
numPos=0;
colsPerSection = 4;
z = length(dToSave(1,:))/colsPerSection;
subplot(2,1,1),  cla;
subplot(2,1,2),  cla;

%Display data
 for i = 2:4:(z*colsPerSection)-2
    y = dToSave(:,i);
    %Strip out NAN's
    nanIndex=isnan(y);
    ind = length(y);
    for j = 1:ind
        if nanIndex(j) == true 
            ind = j-1;
            break
        end
    end
    %Read reduced data set
    y = dToSave(1:ind,i);
    x = dToSave(1:ind,i+1);
    %Plot curve
    if mean(y) > 0%overallMean %Positive
        subplot(2,1,1),  hold on, plot(x,y, 'go' ), hold off;
        posValuesY=[posValuesY ; y];
        posValuesX=[posValuesX; x];
        numPos = numPos + 1;
    else
        subplot(2,1,2),  hold on, plot(x,y, 'go' ), hold off;
        negValuesY=[negValuesY ; y];
        negValuesX=[negValuesX; x];
        numNeg = numNeg + 1;
    end
end

%Fit curves
%Positive first...
maxXPolyFit=[];
maxYPolyFit=[];
maxParameters=[];
if numPos > 0
    %Get points to fit over
    for i = 1:length(posValuesX)
       if ((posValuesX(i) > minPos ) && (posValuesX(i) < maxPos))
           maxXPolyFit=[maxXPolyFit;posValuesX(i)];
           maxYPolyFit=[maxYPolyFit;posValuesY(i)];
       end
    end
    maxParameters = polyfit(maxXPolyFit,maxYPolyFit,4);
    maxFitLabel = sprintf('Best fit = %f x^4 %f x^3 %+f x^2 %+f x %+f', maxParameters);
    maxXPolyFit=[];
    maxYPolyFit=[];
    %Plot fit
    for i=minPos:1:maxPos
        maxXPolyFit=[maxXPolyFit;i];
        xVal= i;
        yVal = (maxParameters(1)* xVal^4) + (maxParameters(2)* xVal^3) + (maxParameters(3)*xVal^2)...
             + (maxParameters(4)*xVal) + maxParameters(5);
        maxYPolyFit=[maxYPolyFit; yVal];
    end
    subplot(2,1,1),  hold on, plot(maxXPolyFit,maxYPolyFit, 'kx-' ), hold off;
    xlabel('Angle (Degrees)');    ylabel('Torque (N m)');
    if strcmp(dType,'Concentric')
        type= ' extension';
    else
        type = ' flexion'
    end
    [p,i]=max(maxYPolyFit);
    s = sprintf('   Maximum%s torque = % .4f  Nm at %.1f degrees\n',  type, p, maxXPolyFit(i));
    title( strcat(maxFitLabel, s));
end
%Then negative
minXPolyFit=[];
minYPolyFit=[];
minParameters=[];
if numNeg > 0
     %Get points to fit over
    for i = 1:length(negValuesX)
       if ((negValuesX(i) > minNeg ) && (negValuesX(i) < maxNeg))
           minXPolyFit=[minXPolyFit;negValuesX(i)];
           minYPolyFit=[minYPolyFit;negValuesY(i)];
       end
    end
    minParameters = polyfit(minXPolyFit,minYPolyFit,4);
    maxFitLabel = sprintf('Best fit = %f x^4 %f x^3 %+f x^2 %+f x %+f', minParameters);
    minXPolyFit=[];
    minYPolyFit=[];
    for i=minNeg: 1: maxNeg
        minXPolyFit=[minXPolyFit;i];
        xVal= i;
        yVal = (minParameters(1)* xVal^4) + (minParameters(2)* xVal^3) + (minParameters(3)*xVal^2)...
            + (minParameters(4)*xVal) + minParameters(5);
        minYPolyFit=[minYPolyFit; yVal];
    end
    subplot(2,1,2),hold on, plot(minXPolyFit,minYPolyFit, 'kx-' ), hold off;
    xlabel('Angle (Degrees)');    ylabel('Torque (N m)');
    if strcmp(dType,'Eccentric')
        type= ' extension';
    else
        type = ' flexion';
    end
    [p,i]=max(abs((minYPolyFit)));
    s = sprintf('   Maximum%s torque = % .4f  Nm at %.1f degrees\n',  type, p, minXPolyFit(i));
    title( strcat(maxFitLabel, s));
end

 set(findobj('Tag','buttonManualSelect'),'Enable', 'on');
 set(findobj('Tag','buttonSelectAll'),'Enable', 'on');
 set(findobj('Tag','buttonDisplay'),'Enable', 'on');
 set(findobj('Tag','buttonSave'),'Enable', 'on');
 set(findobj('Tag','buttonCurveFit'),'Enable', 'on');
 set(findobj('Tag','buttonSaveFit'),'Enable', 'on');

% --- Executes on button press in buttonSelectAll.
function buttonSelectAll_Callback(hObject, eventdata, handles)
% hObject    handle to buttonSelectAll (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

selectAll;

function selectAll


%Selects all data meeting the selection criteria

global d; 
global numSections;
global useSection;
global sectionPoints;
global dProcessed;
global maxTorqueValues;
global minTorqueValues;
global maxTimeValues;
global minTimeValues;
global maxVelocityValues;
global minVelocityValues;
global maxAngleValues;
global minAngleValues;
global TIMECOL;
global TORQUECOL;
global ANGLECOL;
global VELCOL;

maxLength = 0;
maxTorqueValues = [];
minTorqueValues=[];
maxTimeValues=[];
minTimeValues=[];
maxVelocityValues=[];
minVelocityValues=[];
maxAngleValues=[];
minAngleValues=[];
numNegRegions = 0;
numPosRegions = 0;

for i = 1:2:numSections*2
    temp = sectionPoints(i+1) - sectionPoints(i) + 1;
    if temp > maxLength
        maxLength= temp;
    end
    y = dProcessed(sectionPoints(i):sectionPoints(i+1),TORQUECOL);
    x = dProcessed(sectionPoints(i):sectionPoints(i+1),TIMECOL);
    p = dProcessed(sectionPoints(i):sectionPoints(i+1),ANGLECOL);
    v = dProcessed(sectionPoints(i):sectionPoints(i+1),VELCOL);
    %Max or min?
    if mean(y) >= 1
        [val, index] = max(y);
        maxTimeValues=[maxTimeValues;x(index)];
        maxVelocityValues=[maxVelocityValues;v(index)];
        maxAngleValues=[maxAngleValues;p(index)];
        maxTorqueValues =[maxTorqueValues; max(y)];
        numPosRegions = numPosRegions + 1;
    else
        [val, index] = min(y);
        minTorqueValues =[minTorqueValues; min(y)];
        minTimeValues=[minTimeValues;x(index)];
        minVelocityValues=[minVelocityValues;v(index)];
        minAngleValues=[minAngleValues;p(index)];
        numNegRegions = numNegRegions + 1;
    end
end

% Select all regions for analysis
useSection = ones(numSections,1);
subplot(3,1,1), cla, hold on, plot(dProcessed(:, TIMECOL),dProcessed(:, TORQUECOL),'co' ), plot(d(:,TIMECOL), d(:,TORQUECOL)), hold off;
subplot(3,1,2), cla, hold on, plot(dProcessed(:, TIMECOL),dProcessed(:, ANGLECOL),'co' ), plot(d(:,TIMECOL), d(:,ANGLECOL)), hold off;
subplot(3,1,3), cla, hold on, plot(dProcessed(:, TIMECOL),dProcessed(:, VELCOL),'co' ), plot(d(:,TIMECOL), d(:,VELCOL)), hold off;
set(findobj('Tag','editNumPosRegions'),'String', num2str(numPosRegions));
set(findobj('Tag','editNumNegRegions'),'String', num2str(numNegRegions));

saveSelectedDataToMemory(maxLength, numSections);
set(findobj('Tag','buttonManualSelect'),'Enable', 'on');
set(findobj('Tag','buttonSelectAll'),'Enable', 'on');
set(findobj('Tag','buttonDisplay'),'Enable', 'on');
set(findobj('Tag','buttonSave'),'Enable', 'on');
set(findobj('Tag','buttonCurveFit'),'Enable', 'on');
set(findobj('Tag','buttonSaveFit'),'Enable', 'off');




function editWeight_Callback(hObject, eventdata, handles)
% hObject    handle to editWeight (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of editWeight as text
%        str2double(get(hObject,'String')) returns contents of editWeight as a double


% --- Executes during object creation, after setting all properties.
function editWeight_CreateFcn(hObject, eventdata, handles)
% hObject    handle to editWeight (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end



function editAngle_Callback(hObject, eventdata, handles)
% hObject    handle to editAngle (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of editAngle as text
%        str2double(get(hObject,'String')) returns contents of editAngle as a double


% --- Executes during object creation, after setting all properties.
function editAngle_CreateFcn(hObject, eventdata, handles)
% hObject    handle to editAngle (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end


% --- Executes on button press in checkGravity.
function checkGravity_Callback(hObject, eventdata, handles)
% hObject    handle to checkGravity (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of checkGravity
% if get(findobj('Tag','checkGravity'),'Value') ==1 
%     set(findobj('Tag','editWeight'),'Enable', 'on');
%     set(findobj('Tag','editAngle'),'Enable', 'on');
% else
%     set(findobj('Tag','editWeight'),'Enable', 'off');
%     set(findobj('Tag','editAngle'),'Enable', 'off');
% end
if get(findobj('Tag','checkGravity'),'Value') ==1 
    set(findobj('Tag','editWeight'),'Enable', 'on');
    set(findobj('Tag','editAngle'),'Enable', 'on');
else
    set(findobj('Tag','editWeight'),'Enable', 'off');
    set(findobj('Tag','editAngle'),'Enable', 'off');
end
%handles.checkGravity.Value



function editMinAngleNeg_Callback(hObject, eventdata, handles)
% hObject    handle to editMinAngleNeg (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of editMinAngleNeg as text
%        str2double(get(hObject,'String')) returns contents of editMinAngleNeg as a double


% --- Executes during object creation, after setting all properties.
function editMinAngleNeg_CreateFcn(hObject, eventdata, handles)
% hObject    handle to editMinAngleNeg (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end



function editMaxAngleNeg_Callback(hObject, eventdata, handles)
% hObject    handle to editMaxAngleNeg (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of editMaxAngleNeg as text
%        str2double(get(hObject,'String')) returns contents of editMaxAngleNeg as a double


% --- Executes during object creation, after setting all properties.
function editMaxAngleNeg_CreateFcn(hObject, eventdata, handles)
% hObject    handle to editMaxAngleNeg (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end



function editMinAnglePos_Callback(hObject, eventdata, handles)
% hObject    handle to editMinAnglePos (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of editMinAnglePos as text
%        str2double(get(hObject,'String')) returns contents of editMinAnglePos as a double


% --- Executes during object creation, after setting all properties.
function editMinAnglePos_CreateFcn(hObject, eventdata, handles)
% hObject    handle to editMinAnglePos (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end



function editMaxAnglePos_Callback(hObject, eventdata, handles)
% hObject    handle to editMaxAnglePos (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of editMaxAnglePos as text
%        str2double(get(hObject,'String')) returns contents of editMaxAnglePos as a double


% --- Executes during object creation, after setting all properties.
function editMaxAnglePos_CreateFcn(hObject, eventdata, handles)
% hObject    handle to editMaxAnglePos (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end


% --- If Enable == 'on', executes on mouse press in 5 pixel border.
% --- Otherwise, executes on mouse press in 5 pixel border or over editMax.
function editMax_ButtonDownFcn(hObject, eventdata, handles)
% hObject    handle to editMax (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)


% --- Executes on button press in buttonSaveFit.
function buttonSaveFit_Callback(hObject, eventdata, handles)
% hObject    handle to buttonSaveFit (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)
saveData(handles);

function saveData(handles)

global baseDirectory;
global dType;

%Saves data to three files: a basic data file, a polyfit file and an
%averaged data file

%Get file name to save raw data to first
[p,filename,e] = fileparts( get(findobj('Tag','filenameEdit'),'String'));
[filename,PathName] = uiputfile('*.csv;', 'Save file name',strcat(baseDirectory,filename));
if filename == 0 
    return;
else
    baseDirectory = PathName;
end
%Write basic header
fid = fopen(strcat(baseDirectory,filename), 'w');
writeBasicHeader(fid);
%Write data details
if strcmp(dType, 'Concentric')
    pos = 'extension';
    neg = 'flexion';
else
    pos = 'flexion';
    neg = 'extension';
end
%Write data
writeData(fid);
fclose(fid);

%Save fit data to separate files
savePolyFit(PathName,filename);
saveAveragedFit(PathName, filename);

function savePolyFit(PathName, filename)

%Creates polyfit data (_poly.csv) file

global maxXPolyFit;
global minXPolyFit;
global maxYPolyFit;
global minYPolyFit;
global maxParameters;
global minParameters;
global dType;

filename = strcat(strtok(filename, '.'), '_poly.csv');
fid = fopen(strcat(PathName, filename), 'w');
%Write polyfit details to file
if strcmp(dType, 'Concentric')
    pos = 'extension';
    neg = 'flexion';
else
    pos = 'flexion';
    neg = 'extension';
end
fprintf(fid, 'Quartic polynomial fit parameters for %s,', pos );
for i =1:length(maxParameters)
    fprintf(fid, ' %.16f, ', maxParameters(i));
end
fprintf(fid, '\n');
fprintf(fid, 'Quartic polynomial fit parameters for %s,', neg );
for i =1:length(minParameters)
    fprintf(fid, ' %.16f, ', minParameters(i));
end
fprintf(fid, '\n');
[p,i]=max(maxYPolyFit);
fprintf(fid, 'Peak fitted %s torque, % .4f \n',  pos, p);
fprintf(fid, 'Peak fitted %s torque angle, % .4f \n',  pos, maxXPolyFit(i));
[p,i]=min(minYPolyFit);
fprintf(fid, 'Peak fitted %s torque, % .4f \n',  neg, abs(p));
fprintf(fid, 'Peak fitted %s torque angle, % .4f \n',  neg, minXPolyFit(i));
fprintf(fid, '%s angle, %s torque, %s angle, %s torque\n', pos, pos, neg, neg);
%Ensure all arrays same length
maxLength = length(minXPolyFit);
if length(maxXPolyFit)>length(minXPolyFit)
    maxLength = length(maxXPolyFit);
    for j= length(minXPolyFit):1:maxLength-1
        minXPolyFit=vertcat(minXPolyFit, NaN);
        minYPolyFit=vertcat(minYPolyFit, NaN);
    end
elseif length(maxXPolyFit)<length(minXPolyFit)
    maxLength = length(minXPolyFit);
    for j= length(maxXPolyFit):1:maxLength-1
        maxXPolyFit=vertcat(maxXPolyFit, NaN);
        maxYPolyFit=vertcat(maxYPolyFit, NaN);
    end
end
%Write fitted values
for i = 1:1:maxLength
   fprintf(fid, '%f,%f,%f,%f \n' , maxXPolyFit(i), maxYPolyFit(i),minXPolyFit(i), minYPolyFit(i));
end
fclose(fid);

function saveAveragedFit(PathName, filename)

%Creates averaged (_aver.csv) data file

global maxXAveragedFit;
global minXAveragedFit;
global maxYAveragedFit;
global minYAveragedFit;
global smoothVal;
global dType;

%Write averaged fit details to file
filename = strcat(strtok(filename, '.'), '_aver.csv');
fid = fopen(strcat(PathName, filename), 'w');

if strcmp(dType, 'Concentric')
    pos = 'extension';
    neg = 'flexion';
else
    pos = 'flexion';
    neg = 'extension';
end
fprintf(fid, 'Smoothed data points, %d \n', smoothVal );
[p,i]=max(maxYAveragedFit);
fprintf(fid, 'Peak fitted %s torque, % .4f \n',  pos, p);
fprintf(fid, 'Peak fitted %s torque angle, % .4f \n',  pos, maxXAveragedFit(i));
[p,i]=min(minYAveragedFit);
fprintf(fid, 'Peak fitted %s torque, % .4f \n',  neg, abs(p));
fprintf(fid, 'Peak fitted %s torque angle, % .4f \n',  neg, minXAveragedFit(i));
fprintf(fid, '\n');
fprintf(fid, '%s angle, %s torque, %s angle, %s torque\n', pos, pos, neg, neg);
%Ensure all arrays same length
maxLength = length(minXAveragedFit);
if length(maxXAveragedFit)>length(minXAveragedFit)
    maxLength = length(maxXAveragedFit);
    for j= length(minXAveragedFit):1:maxLength-1
        minXAveragedFit=vertcat(minXAveragedFit, NaN);
        minYAveragedFit=vertcat(minYAveragedFit, NaN);
    end
elseif length(maxXAveragedFit)<length(minXAveragedFit)
    maxLength = length(minXAveragedFit);
    for j= length(maxXAveragedFit):1:maxLength-1
        maxXAveragedFit=vertcat(maxXAveragedFit, NaN);
        maxYAveragedFit=vertcat(maxYAveragedFit, NaN);
    end
end
%Write fitted values
for i = 1:1:maxLength
   fprintf(fid, '%f,%f,%f,%f \n' , maxXAveragedFit(i), maxYAveragedFit(i),minXAveragedFit(i), minYAveragedFit(i));
end
fclose(fid);

% --- Executes on slider movement.
function maxVelSlider_Callback(hObject, eventdata, handles)
% hObject    handle to maxVelSlider (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'Value') returns position of slider
%        get(hObject,'Min') and get(hObject,'Max') to determine range of slider

v =  get(findobj('Tag','maxVelSlider'),'Value');
set(findobj('Tag','editMax'),'String', num2str(v)); 
drawnow;

% --- Executes during object creation, after setting all properties.
function maxVelSlider_CreateFcn(hObject, eventdata, handles)
% hObject    handle to maxVelSlider (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: slider controls usually have a light gray background.
if isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor',[.9 .9 .9]);
end


% --- If Enable == 'on', executes on mouse press in 5 pixel border.
% --- Otherwise, executes on mouse press in 5 pixel border or over maxVelSlider.
function maxVelSlider_ButtonDownFcn(hObject, eventdata, handles)
% hObject    handle to maxVelSlider (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)




% --- Executes on button press in buttonAutoFit.
function buttonAutoFit_Callback(hObject, eventdata, handles)
% hObject    handle to buttonAutoFit (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

%Performs polynomial and averagd data fit to selected torque vs angle
%values

selectAll;
if get(handles.radioAverage,'Value') == 1 
    polyCurveFitting;
    averagedCurveFitting(handles);
else
    averagedCurveFitting(handles);
    polyCurveFitting;
end 



function [failed] = autoSelectConcentricData(velMin, velMax, velTol)

%Performs automated selection of concentric data sets

global dProcessed;
global numSections;
global sectionPoints;
global TORQUECOL;
global ANGLECOL;
global VELCOL;

failed =0;

%Do basic processing according to velocity limits
applyVelocityFiltering(velMin, velMax, velTol);
createSectionData;
displayRawData;

%Do next stage of filtering according to specified range of angle motion, 
rangeOfMotion=str2num(get(findobj('Tag','editROM'),'String'));
dDummy =dProcessed;
for i = numSections*2:-2:1
    if abs(dProcessed(sectionPoints(i-1), ANGLECOL) - dProcessed(sectionPoints(i), ANGLECOL)) < rangeOfMotion 
        dDummy(sectionPoints(i-1):sectionPoints(i),:)=[];
     end
end
if ~isempty(dDummy)
    dProcessed = dDummy;
else
    failed =1;
    return;
end
%displaySelectedData;
createSectionData;    

%Do angle filtering too - ignore first nad last n degrees of movement
omitROMFirst = str2num(get(findobj('Tag','editROMOmitFirst'),'String'));
omitROMLast = str2num(get(findobj('Tag','editROMOmitLast'),'String'));
dDummy = dProcessed;
maxAngle =max(dProcessed(:,ANGLECOL));
minAngle = min(dProcessed(:,ANGLECOL));
for i = length(dProcessed):-1:1
     if dProcessed(i,VELCOL) > 0
        if dProcessed(i,ANGLECOL) > (maxAngle - omitROMFirst) || dProcessed(i,ANGLECOL) < (minAngle + omitROMLast)
            dDummy(i,:)=[];
        end
    else
        if dProcessed(i,ANGLECOL) < (minAngle + omitROMFirst) || dProcessed(i,ANGLECOL) > (maxAngle - omitROMLast)
            dDummy(i,:)=[];
        end
    end
end
if ~isempty(dDummy)
    dProcessed = dDummy;
else
    failed =1;
    return;
end
%displaySelectedData;
createSectionData;

dDummy = dProcessed;
%Remove false datasets where vel pos and toque neg, and vice versa
for i = length(dProcessed):-1:1
    if (dProcessed(i,VELCOL) < 0 && dProcessed(i,TORQUECOL) >= 0) ...
            || (dProcessed(i,VELCOL) > 0 && dProcessed(i,TORQUECOL) <= 0)
        dDummy(i,:)=[];
    end
end
if ~isempty(dDummy)
    dProcessed = dDummy;
else
    failed =1;
    return;
end
%displaySelectedData;
createSectionData;

%Filter according to torque next. Check max and min peak torques of each section within 10% of max value,
%discard section if not.
%Separate sections into quadriceps (neg torque) and hamstrings (pos
%torque)and do each one seperatley
torTol= (1-str2num(get(findobj('Tag','editTorTol'),'String'))/100);
romTol= (1-str2num(get(findobj('Tag','editROMTol'),'String'))/100);
hamIndices = []; 
for i = 1:2:numSections*2;
    if (dProcessed(sectionPoints(i),ANGLECOL ) < dProcessed(sectionPoints(i+1),ANGLECOL )) 
        hamIndices = [hamIndices sectionPoints(i) sectionPoints(i+1)];
    end
end
%find max and min and indices
[maxTorque maxIndex]= max(dProcessed(:,TORQUECOL));
[minTorque minIndex]= min(dProcessed(:,TORQUECOL));
%Check max and min peak torques of each section within 10% of max value,
%discard section if not
dDummy =dProcessed;
for i = length(hamIndices):-2:1
    if min(dProcessed(hamIndices(i-1):hamIndices(i), TORQUECOL)) > torTol* minTorque 
        dDummy(hamIndices(i-1):hamIndices(i),:)=[];
        hamIndices(i)= [];
        hamIndices(i-1) =[];
     end
end

%Filter according to  range of motion wrt max torque peak.
%I.e. Check range of motion (angle) of other section is within toleranace of max section
%Get new max and min indices first
[maxTorque maxIndex]= max(dDummy(:,TORQUECOL));
[minTorque minIndex]= min(dDummy(:,TORQUECOL));
%Get ROM for max hamstring peak
maxROMHam =0;
for i = length(hamIndices):-2:1
    if hamIndices(i-1)< minIndex && hamIndices(i)> minIndex
        maxROMHam = abs(dProcessed(hamIndices(i),ANGLECOL)- dProcessed(hamIndices(i-1),ANGLECOL));
        break;
    end
end   
%Discard sections where outside tolerance wrt max ROM
for i = length(hamIndices):-2:1
    if abs(dProcessed(hamIndices(i),ANGLECOL)- dProcessed(hamIndices(i-1),ANGLECOL)) < romTol* maxROMHam
        dDummy(hamIndices(i-1):hamIndices(i),:)=[];
        hamIndices(i)= [];
        hamIndices(i-1) =[];
    end
end     
%Set limits for fitting curves
set(findobj('Tag','editMinAngleNeg'), 'String', num2str(min(dProcessed(hamIndices,ANGLECOL))));
set(findobj('Tag','editMaxAngleNeg'), 'String', num2str(max(dProcessed(hamIndices,ANGLECOL))));
dProcessed = dDummy;

%Repeat for qauds
createSectionData;    
quadIndices = []; 
for i = 1:2:numSections*2;
    if (dProcessed(sectionPoints(i),ANGLECOL ) > dProcessed(sectionPoints(i+1),ANGLECOL )) 
        quadIndices = [quadIndices sectionPoints(i) sectionPoints(i+1)];
    end
end
%find max and min and indices
[maxTorque maxIndex]= max(dProcessed(:,TORQUECOL));
[minTorque minIndex]= min(dProcessed(:,TORQUECOL));
%Check max and min peak torques of each section within tolerance of max value,
%discard section if not
dDummy =dProcessed;
for i = length(quadIndices):-2:1
    if max(dProcessed(quadIndices(i-1):quadIndices(i), TORQUECOL)) < torTol* maxTorque 
        dDummy(quadIndices(i-1):quadIndices(i),:)=[];
        quadIndices(i)= [];
        quadIndices(i-1) =[];
     end
end

%Filter according to  range of motion wrt max torque peak.
%I.e. Check range of motion (angle) of other section is within 10% of max section
%Get new max and min indices first
[maxTorque maxIndex]= max(dDummy(:,TORQUECOL));
[minTorque minIndex]= min(dDummy(:,TORQUECOL));
%Get ROM for max quads peak
maxROMQuad =0; 
for i = length(quadIndices):-2:1
    if quadIndices(i-1)< maxIndex && quadIndices(i) > maxIndex
        maxROMQuad = abs(dProcessed(quadIndices(i),ANGLECOL)- dProcessed(quadIndices(i-1),ANGLECOL));
        break;
    end
end    
%Discard sections where outside tolerance wrt max ROM
for i = length(quadIndices):-2:1
    if abs(dProcessed(quadIndices(i),ANGLECOL)- dProcessed(quadIndices(i-1),ANGLECOL)) < romTol* maxROMQuad
        dDummy(quadIndices(i-1):quadIndices(i),:)=[];
        quadIndices(i)= [];
        quadIndices(i-1) =[];
    end
end     
%Set limits for fitting curves
set(findobj('Tag','editMinAnglePos'), 'String', num2str(min(dProcessed(quadIndices,ANGLECOL))));
set(findobj('Tag','editMaxAnglePos'), 'String', num2str(max(dProcessed(quadIndices,ANGLECOL))));
dProcessed = dDummy;

displaySelectedData;
createSectionData;

drawnow;

function [failed] = autoSelectEccentricData(velMin, velMax, velTol)

%Performs automated selection of eccentric data sets

global dProcessed;
global numSections;
global sectionPoints;
global TORQUECOL;
global ANGLECOL;
global VELCOL;

failed =0;
%Do basic processing according to velocity limits
applyVelocityFiltering(velMin, velMax, velTol);
createSectionData;
displayRawData;

%Do next stage of filtering according to min range of angle motion, 
rangeOfMotion=str2num(get(findobj('Tag','editROM'),'String'));
dDummy =dProcessed;
for i = numSections*2:-2:1
    if abs(dProcessed(sectionPoints(i-1), ANGLECOL) - dProcessed(sectionPoints(i), ANGLECOL)) < rangeOfMotion 
        dDummy(sectionPoints(i-1):sectionPoints(i),:)=[];
     end
end
if ~isempty(dDummy)
    dProcessed = dDummy;
else
    failed =1;
    return;
end
%displaySelectedData;
createSectionData;    

%Do angle filtering too - ignore first  and lst n degrees of movement
omitROMFirst = str2num(get(findobj('Tag','editROMOmitFirst'),'String'));
omitROMLast = str2num(get(findobj('Tag','editROMOmitLast'),'String'));
dDummy = dProcessed;
maxAngle =max(dProcessed(:,ANGLECOL));
minAngle = min(dProcessed(:,ANGLECOL));
for i = length(dProcessed):-1:1
     if dProcessed(i,VELCOL) > 0
        if dProcessed(i,ANGLECOL) > (maxAngle - omitROMFirst) || dProcessed(i,ANGLECOL) < (minAngle + omitROMLast)
            dDummy(i,:)=[];
        end
    else
        if dProcessed(i,ANGLECOL) < (minAngle + omitROMFirst) || dProcessed(i,ANGLECOL) > (maxAngle - omitROMLast)
            dDummy(i,:)=[];
        end
    end
end
if ~isempty(dDummy)
    dProcessed = dDummy;
else
    failed =1;
    return;
end
%displaySelectedData;

%Remove false datasets i.e. velocity <0
for i = length(dProcessed):-1:1
%     if (dProcessed(i,VELCOL) < 0 && dProcessed(i,TORQUECOL) >= 0) ...
%             || (dProcessed(i,VELCOL) > 0 && dProcessed(i,TORQUECOL) <= 0) ...
    if dProcessed(i,VELCOL) < 0
        dDummy(i,:)=[];
    end
end
if ~isempty(dDummy)
    dProcessed = dDummy;
else
    failed =1;
    return;
end
%displaySelectedData;
createSectionData;

%Filter according to torque next. Check max and min peak torques of each section within 10% of max value,
%discard section if not.
hamIndices = sectionPoints; 
%find min torque and index
[minTorque minIndex]= min(dProcessed(:,TORQUECOL));
%Check min peak torques of each section within specified tolerance of min value,
%discard section if not
torTol= (1-str2num(get(findobj('Tag','editTorTol'),'String'))/100);
dDummy =dProcessed;
for i = length(hamIndices):-2:1
    if min(dProcessed(hamIndices(i-1):hamIndices(i), TORQUECOL)) > torTol * minTorque 
        dDummy(hamIndices(i-1):hamIndices(i),:)=[];
        hamIndices(i)= [];
        hamIndices(i-1) =[];
     end
end

%Filter according to range of motion wrt max torque peak.
%I.e. Check range of motion (angle) of other section is within 10% of max section
%Get new min index first
[minTorque minIndex]= min(dDummy(:,TORQUECOL));
%Get ROM for max hamstring peak
maxROMHam =0;
for i = length(hamIndices):-2:1
    if hamIndices(i-1)< minIndex && hamIndices(i)> minIndex
        maxROMHam = abs(dProcessed(hamIndices(i),ANGLECOL)- dProcessed(hamIndices(i-1),ANGLECOL));
        break;
    end
end   
%Discard sections where outside tolerance wrt max ROM
romTol= (1-str2num(get(findobj('Tag','editROMTol'),'String'))/100);
for i = length(hamIndices):-2:1
    if abs(dProcessed(hamIndices(i),ANGLECOL)- dProcessed(hamIndices(i-1),ANGLECOL)) < romTol* maxROMHam
        dDummy(hamIndices(i-1):hamIndices(i),:)=[];
        hamIndices(i)= [];
        hamIndices(i-1) =[];
    end
end     

%Set limits for fitting curves
set(findobj('Tag','editMinAngleNeg'), 'String', num2str(min(dProcessed(hamIndices,ANGLECOL))));
set(findobj('Tag','editMaxAngleNeg'), 'String', num2str(max(dProcessed(hamIndices,ANGLECOL))));
dProcessed = dDummy;
displaySelectedData;
createSectionData;

drawnow;


function editVelTol_Callback(hObject, eventdata, handles)
% hObject    handle to editVelTol (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of editVelTol as text
%        str2double(get(hObject,'String')) returns contents of editVelTol as a double


% --- Executes during object creation, after setting all properties.
function editVelTol_CreateFcn(hObject, eventdata, handles)
% hObject    handle to editVelTol (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end



function editROM_Callback(hObject, eventdata, handles)
% hObject    handle to editROM (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of editROM as text
%        str2double(get(hObject,'String')) returns contents of editROM as a double


% --- Executes during object creation, after setting all properties.
function editROM_CreateFcn(hObject, eventdata, handles)
% hObject    handle to editROM (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end



function editROMOmitFirst_Callback(hObject, eventdata, handles)
% hObject    handle to editROMOmitFirst (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of editROMOmitFirst as text
%        str2double(get(hObject,'String')) returns contents of editROMOmitFirst as a double


% --- Executes during object creation, after setting all properties.
function editROMOmitFirst_CreateFcn(hObject, eventdata, handles)
% hObject    handle to editROMOmitFirst (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end



function editROMTol_Callback(hObject, eventdata, handles)
% hObject    handle to editROMTol (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of editROMTol as text
%        str2double(get(hObject,'String')) returns contents of editROMTol as a double


% --- Executes during object creation, after setting all properties.
function editROMTol_CreateFcn(hObject, eventdata, handles)
% hObject    handle to editROMTol (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end



function editTorTol_Callback(hObject, eventdata, handles)
% hObject    handle to editTorTol (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of editTorTol as text
%        str2double(get(hObject,'String')) returns contents of editTorTol as a double


% --- Executes during object creation, after setting all properties.
function editTorTol_CreateFcn(hObject, eventdata, handles)
% hObject    handle to editTorTol (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end


% --- Executes on button press in buttonAutoSelect.
function buttonAutoSelect_Callback(hObject, eventdata, handles)
% hObject    handle to buttonAutoSelect (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)


%Automatically selects data according to criteria displayed on the GUI

global dType;

velMax = str2num(get(findobj('Tag', 'editMaxVelocity'), 'String'));
velMin = str2num(get(findobj('Tag', 'editMinVelocity'), 'String'));
velTol = str2num(get(findobj('Tag', 'editVelTol'), 'String'))/100;
if strcmp(dType,'Eccentric')
    failed = autoSelectEccentricData(velMin, velMax, velTol);
else
    failed = autoSelectConcentricData(velMin, velMax, velTol);
end
if ~failed 
    set(findobj('Tag','buttonSave'),'Enable', 'on');
    set(findobj('Tag','buttonCurveFit'),'Enable', 'on');
    set(findobj('Tag','buttonSaveFit'),'Enable', 'on');
    selectAll;
else
    set(findobj('Tag','editNumPosRegions'),'String', '0');
    set(findobj('Tag','editNumNegRegions'),'String', '0');
end


function editROMOmitLast_Callback(hObject, eventdata, handles)
% hObject    handle to editROMOmitLast (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of editROMOmitLast as text
%        str2double(get(hObject,'String')) returns contents of editROMOmitLast as a double


% --- Executes during object creation, after setting all properties.
function editROMOmitLast_CreateFcn(hObject, eventdata, handles)
% hObject    handle to editROMOmitLast (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end


% --- Executes on button press in radioAverage.
function radioAverage_Callback(hObject, eventdata, handles)
% hObject    handle to radioAverage (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of radioAverage

if get(handles.radioAverage, 'Value') == 1
    set(handles.editSmooth, 'Enable', 'on');
    set(handles.textSmooth, 'Enable', 'on');
else
    set(handles.editSmooth, 'Enable', 'off');
    set(handles.textSmooth, 'Enable', 'off');
end

function editSmooth_Callback(hObject, eventdata, handles)
% hObject    handle to editSmooth (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of editSmooth as text
%        str2double(get(hObject,'String')) returns contents of editSmooth as a double


% --- Executes during object creation, after setting all properties.
function editSmooth_CreateFcn(hObject, eventdata, handles)
% hObject    handle to editSmooth (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end


% --- Executes on button press in radioPoly.
function radioPoly_Callback(hObject, eventdata, handles)
% hObject    handle to radioPoly (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of radioPoly
if get(handles.radioPoly, 'Value') == 1
    set(handles.editSmooth, 'Enable', 'off');
    set(handles.textSmooth, 'Enable', 'off');
else
    set(handles.editSmooth, 'Enable', 'on');
    set(handles.textSmooth, 'Enable', 'on');
end

function [c,ww] = smooth(varargin)
%SMOOTH  Smooth data.
%   Z = SMOOTH(Y) smooths data Y using a 5-point moving average.
%
%   Z = SMOOTH(Y,SPAN) smooths data Y using SPAN as the number of points used
%   to compute each element of Z.
%
%   Z = SMOOTH(Y,SPAN,METHOD) smooths data Y with specified METHOD. The
%   available methods are:
%
%           'moving'   - Moving average (default)
%           'lowess'   - Lowess (linear fit)
%           'loess'    - Loess (quadratic fit)
%           'sgolay'   - Savitzky-Golay
%           'rlowess'  - Robust Lowess (linear fit)
%           'rloess'   - Robust Loess (quadratic fit)
%
%   Z = SMOOTH(Y,METHOD) uses the default SPAN 5.
%
%   Z = SMOOTH(Y,SPAN,'sgolay',DEGREE) and Z = SMOOTH(Y,'sgolay',DEGREE)
%   additionally specify the degree of the polynomial to be used in the
%   Savitzky-Golay method. The default DEGREE is 2. DEGREE must be smaller
%   than SPAN.
%
%   Z = SMOOTH(X,Y,...) additionally specifies the X coordinates.  If X is
%   not provided, methods that require X coordinates assume X = 1:N, where
%   N is the length of Y.
%
%   Notes:
%   1. When X is given and X is not uniformly distributed, the default method
%   is 'lowess'.  The 'moving' method is not recommended.
%
%   2. For the 'moving' and 'sgolay' methods, SPAN must be odd.
%   If an even SPAN is specified, it is reduced by 1.
%
%   3. If SPAN is greater than the length of Y, it is reduced to the
%   length of Y.
%
%   4. In the case of (robust) lowess and (robust) loess, it is also
%   possible to specify the SPAN as a percentage of the total number
%   of data points. When SPAN is less than or equal to 1, it is
%   treated as a percentage.
%
%   For example:
%
%   Z = SMOOTH(Y) uses the moving average method with span 5 and
%   X=1:length(Y).
%
%   Z = SMOOTH(Y,7) uses the moving average method with span 7 and
%   X=1:length(Y).
%
%   Z = SMOOTH(Y,'sgolay') uses the Savitzky-Golay method with DEGREE=2,
%   SPAN = 5, X = 1:length(Y).
%
%   Z = SMOOTH(X,Y,'lowess') uses the lowess method with SPAN=5.
%
%   Z = SMOOTH(X,Y,SPAN,'rloess') uses the robust loess method.
%
%   Z = SMOOTH(X,Y) where X is unevenly distributed uses the
%   'lowess' method with span 5.
%
%   Z = SMOOTH(X,Y,8,'sgolay') uses the Savitzky-Golay method with
%   span 7 (8 is reduced by 1 to make it odd).
%
%   Z = SMOOTH(X,Y,0.3,'loess') uses the loess method where span is
%   30% of the data, i.e. span = ceil(0.3*length(Y)).
%
%   See also SPLINE.

%   Copyright 2001-2005 The MathWorks, Inc.
%   $Revision: 1.17.4.9 $  $Date: 2005/06/21 19:20:49 $

if nargin < 1
    error('curvefit:smooth:needMoreArgs', ...
        'SMOOTH needs at least one argument.');
end

if nargout > 1 % Called from the GUI cftool
    ws = warning('off', 'all'); % turn warning off and record the previous warning state.
    [lw,lwid] = lastwarn;
    lastwarn('');
else
    ws = warning('query','all'); % Leave warning state alone but save it so resets are no-ops.
end

% is x given as the first argument?
if nargin==1 || ( nargin > 1 && (length(varargin{2})==1 || ischar(varargin{2})) )
    % smooth(Y) | smooth(Y,span,...) | smooth(Y,method,...)
    is_x = 0; % x is not given
    y = varargin{1};
    y = y(:);
    x = (1:length(y))';
else % smooth(X,Y,...)
    is_x = 1;
    y = varargin{2};
    x = varargin{1};
    y = y(:);
    x = x(:);
end

% is span given?
span = [];
if nargin == 1+is_x || ischar(varargin{2+is_x})
    % smooth(Y), smooth(X,Y) || smooth(X,Y,method,..), smooth(Y,method)
    is_span = 0;
else
    % smooth(...,SPAN,...)
    is_span = 1;
    span = varargin{2+is_x};
end

% is method given?
method = [];
if nargin >= 2+is_x+is_span
    % smooth(...,Y,method,...) | smooth(...,Y,span,method,...)
    method = varargin{2+is_x+is_span};
end

t = length(y);
if t == 0
    c = y;
    ww = '';
    if nargout > 1
        ww = lastwarn;
        lastwarn(lw,lwid);
        warning(ws);  % turn warning back to the previous state.
    end
    return
elseif length(x) ~= t
    warning(ws); % reset warn state before erroring
    error('curvefit:smooth:XYmustBeSameLength',...
        'X and Y must be the same length.');
end

if isempty(method)
    diffx = diff(x);
    if uniformx(diffx,x,y)
        method = 'moving'; % uniformly distributed X.
    else
        method = 'lowess';
    end
end

% realize span
if span <= 0
    warning(ws); % reset warn state before erroring
    error('curvefit:smooth:spanMustBePositive', ...
        'SPAN must be positive.');
end
if span < 1, span = ceil(span*t); end % percent convention
if isempty(span), span = 5; end % smooth(Y,[],method)

idx = 1:t;

sortx = any(diff(isnan(x))<0);   % if NaNs not all at end
if sortx || any(diff(x)<0) % sort x
    [x,idx] = sort(x);
    y = y(idx);
end

c = repmat(NaN,size(y));
ok = ~isnan(x);
switch method
    case 'moving'
        c(ok) = moving(x(ok),y(ok),span);
    case {'lowess','loess','rlowess','rloess'}
        robust = 0;
        iter = 5;
        if method(1)=='r'
            robust = 1;
            method = method(2:end);
        end
        c(ok) = lowess(x(ok),y(ok),span, method,robust,iter);
    case 'sgolay'
        if nargin >= 3+is_x+is_span
            degree = varargin{3+is_x+is_span};
        else
            degree = 2;
        end
        if degree < 0 || degree ~= floor(degree) || degree >= span
            warning(ws); % reset warn state before erroring
            error('curvefit:smooth:invalidDegree', ...
                'Degree must be an integer between 0 and span-1.');
        end
        c(ok) = sgolay(x(ok),y(ok),span,degree);
    otherwise
        warning(ws); % reset warn state before erroring
        error('curvefit:smooth:unrecognizedMethod', ...
            'SMOOTH: Unrecognized method.');
end

c(idx) = c;

if nargout > 1
    ww = lastwarn;
    lastwarn(lw,lwid);
    warning(ws);  % turn warning back to the previous state.
end


function c = moving(x,y, span)
% moving average of the data.

ynan = isnan(y);
span = floor(span);
n = length(y);
span = min(span,n);
width = span-1+mod(span,2); % force it to be odd
xreps = any(diff(x)==0);
if width==1 && ~xreps && ~any(ynan), c = y; return; end
if ~xreps && ~any(ynan)
    % simplest method for most common case
    c = filter(ones(width,1)/width,1,y);
    cbegin = cumsum(y(1:width-2));
    cbegin = cbegin(1:2:end)./(1:2:(width-2))';
    cend = cumsum(y(n:-1:n-width+3));
    cend = cend(end:-2:1)./(width-2:-2:1)';
    c = [cbegin;c(width:end);cend];
elseif ~xreps
    % with no x repeats, can take ratio of two smoothed sequences
    yy = y;
    yy(ynan) = 0;
    nn = double(~ynan);
    ynum = moving(x,yy,span);
    yden = moving(x,nn,span);
    c = ynum ./ yden;
else
    % with some x repeats, loop
    notnan = ~ynan;
    yy = y;
    yy(ynan) = 0;
    c = zeros(n,1);
    for i=1:n
        if i>1 && x(i)==x(i-1)
            c(i) = c(i-1);
            continue;
        end
        R = i;                                 % find rightmost value with same x
        while(R<n && x(R+1)==x(R))
            R = R+1;
        end
        hf = ceil(max(0,(span - (R-i+1))/2));  % need this many more on each side
        hf = min(min(hf,(i-1)), (n-R));
        L = i-hf;                              % find leftmost point needed
        while(L>1 && x(L)==x(L-1))
            L = L-1;
        end
        R = R+hf;                              % find rightmost point needed
        while(R<n && x(R)==x(R+1))
            R = R+1;
        end
        c(i) = sum(yy(L:R)) / sum(notnan(L:R));
    end
end

% -----------------------------------------
function isuniform = uniformx(diffx,x,y)
%ISUNIFORM True if x is of the form a:b:c

if any(isnan(y)) || any(isnan(x))
    isuniform = false;
else
    isuniform = all(abs(diff(diffx)) <= eps*max(abs([x(1),x(end)])));
end


% --- Executes on button press in pushbutton39.
function pushbutton39_Callback(hObject, eventdata, handles)
% hObject    handle to pushbutton39 (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)


% --- Executes on button press in radioBiodex.
function radioBiodex_Callback(hObject, eventdata, handles)
% hObject    handle to radioBiodex (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of radioBiodex


% --- Executes on button press in radioCybex.
function radioCybex_Callback(hObject, eventdata, handles)
% hObject    handle to radioCybex (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of radioCybex


% --- Executes on button press in radioOther.
function radioOther_Callback(hObject, eventdata, handles)
% hObject    handle to radioOther (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of radioOther


% --- If Enable == 'on', executes on mouse press in 5 pixel border.
% --- Otherwise, executes on mouse press in 5 pixel border or over text23.
function text23_ButtonDownFcn(hObject, eventdata, handles)
% hObject    handle to text23 (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)


% --- Executes during object creation, after setting all properties.
function buttonCurveFit_CreateFcn(hObject, eventdata, handles)
% hObject    handle to buttonCurveFit (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called


% --- Executes on button press in importButton.
function importButton_Callback(hObject, eventdata, handles)
% hObject    handle to importButton (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

importFilesToInterpolate;

function importFilesToInterpolate

%Import previously stored fit files and interpolates the torque data
% between the overall min and max angle data. The data is then in a format suitable
% for export to SPS or other statistical software packages and is saved to a csv file

global baseDirectory;

%Get file names to import
[FileNames,PathName] = uigetfile({'*.csv', 'Data Files (*.csv)'; ...
     }, 'Select the files to process', baseDirectory,'MultiSelect','on');
 if ~iscell(FileNames)
    return;
 else
    set(findobj('Tag','selectedFilesText'),'String',FileNames);
    baseDirectory = PathName;
 end
 %Import data
 numFiles = length(FileNames);
 files={numFiles};
 minAng=0; maxAng= 0; oldData=[]; mins=[]; maxs=[];
 for i= 1: numFiles
     [num,txt,raw] = xlsread(strcat(baseDirectory,FileNames{i}));
     if isempty(num)
         %This clause put in because the Belgian version of matlab returns
         %an empty num array. No idea why!?
         offset = 8;
         maxL= length(raw(:,1))-offset+1 ;
         maxW = 4; %length(raw(offset,:));
         store = NaN(maxL,maxW);
         for  k=1:maxL
             s= raw(k + offset -1);
             for j=1:maxW
                 [s1, s] = strtok(s, ',');
                 if  isnumeric(str2num(cell2mat(s1)))
                     store(k, j)=str2num(cell2mat(s1));
                 end
             end
         end
         files{i}=store;
     else
         %files{i} = num(8:end,1:4);
         maxL=length(raw(8:end,1));
         maxW=4;
         store = NaN(maxL,maxW);
         for k=1:maxL
             for j=1:maxW
                if isnumeric(cell2mat(raw(k+7,j)))
                    store(k,j)=cell2mat(raw(k+7,j));
                end 
             end
         end
         files{i}=store;
     end
      maxs = [maxs, max(files{i}(:,1))];
      maxs = [maxs, max(files{i}(:,3))];
      mins = [mins, min(files{i}(:,1))];
      mins = [mins, min(files{i}(:,3))];
 end
 %Find min and max angles from all files
  maxAng = max(maxs);
  minAng = min(mins);
 maxAng = floor(maxAng);
 minAng = ceil(minAng);
 
 %Create appropraitely sized NaN output array
 outputArray = NaN(maxAng-minAng+1 ,numFiles*4);
 %create maskAray of values to interpolate to
 maskArray = [minAng:maxAng]';
 
 %Fill in all possible angle values 
for j = 1:2:numFiles*4
    outputArray(:,j) = maskArray;
end

%Fill in appropriate torque values for each angle value
for j = 1:numFiles
    for k = 1:2:3
        %Create temporary array for x values
        i= 1; tempX =[];
        while ~isnan(files{j}(i,k)) 
            tempX = [tempX files{j}(i,k)];
            i = i +1;
            if i > length(files{j}(:,k))
                break;
            end
        end
        %Create temporary array for y values
        i= 1; tempY=[];
        while ~isnan(files{j}(i,k)) && i <= length(files{j}(:,k))
            tempY = [tempY files{j}(i,k+1)];
            i = i +1;
            if i > length(files{j}(:,k))
                break;
            end
        end
        if ~isempty(tempX)
            %Interpolate x and y arrays to maskArray values
            tempOutput = interp1(tempX,tempY,maskArray);
            %Overwrite output array values with interpolated ones;
            for i =1:length(tempOutput)
                outputArray(i,(j-1)*4+k+1)= tempOutput(i);
            end;
        end
    end
end

 %Save to appropriate csv file
%Get file name to save processed data to first
[filename,PathName] = uiputfile('*.csv;', 'Save file name',baseDirectory);
if filename == 0 
    return;
else 
    %Create and write headers
    baseDirectory = PathName;
    fid = fopen(strcat(baseDirectory, '\',filename), 'w');
    for i = 1:numFiles
        fprintf(fid, '%s %s %s %s %s %s %s %s', FileNames{1,i},',',' ',',',' ',',',' ',',');
    end
    fprintf(fid, '\n');
    for i = 1:numFiles
        fprintf(fid, '%s %s %s %s %s %s %s %s','ext. angle',',','ext. torque',',','flexion angle',',','flexion torque',',');
    end
    fprintf(fid, '\n');
    fclose(fid);
    %Write output array data
    dlmwrite(strcat(baseDirectory, '\',filename), outputArray, '-append');
end
 
function selectedFilesText_Callback(hObject, eventdata, handles)
% hObject    handle to selectedFilesText (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of selectedFilesText as text
%        str2double(get(hObject,'String')) returns contents of selectedFilesText as a double


% --- Executes during object creation, after setting all properties.
function selectedFilesText_CreateFcn(hObject, eventdata, handles)
% hObject    handle to selectedFilesText (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end



function editMaxVelocity_Callback(hObject, eventdata, handles)
% hObject    handle to editMaxVelocity (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of editMaxVelocity as text
%        str2double(get(hObject,'String')) returns contents of editMaxVelocity as a double


% --- Executes during object creation, after setting all properties.
function editMaxVelocity_CreateFcn(hObject, eventdata, handles)
% hObject    handle to editMaxVelocity (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end



function editMinVelocity_Callback(hObject, eventdata, handles)
% hObject    handle to editMinVelocity (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of editMinVelocity as text
%        str2double(get(hObject,'String')) returns contents of editMinVelocity as a double


% --- Executes during object creation, after setting all properties.
function editMinVelocity_CreateFcn(hObject, eventdata, handles)
% hObject    handle to editMinVelocity (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end



function editNumNegRegions_Callback(hObject, eventdata, handles)
% hObject    handle to editNumNegRegions (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of editNumNegRegions as text
%        str2double(get(hObject,'String')) returns contents of editNumNegRegions as a double


% --- Executes during object creation, after setting all properties.
function editNumNegRegions_CreateFcn(hObject, eventdata, handles)
% hObject    handle to editNumNegRegions (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end



function filenameEdit_Callback(hObject, eventdata, handles)
% hObject    handle to filenameEdit (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of filenameEdit as text
%        str2double(get(hObject,'String')) returns contents of filenameEdit as a double


% --- Executes during object creation, after setting all properties.
function filenameEdit_CreateFcn(hObject, eventdata, handles)
% hObject    handle to filenameEdit (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    empty - handles not created until after all CreateFcns called

% Hint: edit controls usually have a white background on Windows.
%       See ISPC and COMPUTER.
if ispc && isequal(get(hObject,'BackgroundColor'), get(0,'defaultUicontrolBackgroundColor'))
    set(hObject,'BackgroundColor','white');
end


% --- Executes on key press with focus on editMax and none of its controls.
function editMax_KeyPressFcn(hObject, eventdata, handles)
% hObject    handle to editMax (see GCBO)
% eventdata  structure with the following fields (see UICONTROL)
%	Key: name of the key that was pressed, in lower case
%	Character: character interpretation of the key(s) that was pressed
%	Modifier: name(s) of the modifier key(s) (i.e., control, shift) pressed
% handles    structure with handles and user data (see GUIDATA)

manualVelocitySelection(eventdata);


% --- Executes on key press with focus on editMin and none of its controls.
function editMin_KeyPressFcn(hObject, eventdata, handles)
% hObject    handle to editMin (see GCBO)
% eventdata  structure with the following fields (see UICONTROL)
%	Key: name of the key that was pressed, in lower case
%	Character: character interpretation of the key(s) that was pressed
%	Modifier: name(s) of the modifier key(s) (i.e., control, shift) pressed
% handles    structure with handles and user data (see GUIDATA)

manualVelocitySelection(eventdata);

% --- Executes on key press with focus on editTol and none of its controls.
function editTol_KeyPressFcn(hObject, eventdata, handles)
% hObject    handle to editTol (see GCBO)
% eventdata  structure with the following fields (see UICONTROL)
%	Key: name of the key that was pressed, in lower case
%	Character: character interpretation of the key(s) that was pressed
%	Modifier: name(s) of the modifier key(s) (i.e., control, shift) pressed
% handles    structure with handles and user data (see GUIDATA)

manualVelocitySelection(eventdata);

function manualVelocitySelection(eventdata)

global d;
global dProcessed;
global TIMECOL;
global ANGLECOL;
global TORQUECOL;
global VELCOL;

if strcmp(eventdata.Key,'return')
    drawnow;
    velMax = str2num(get(findobj('Tag', 'editMax'), 'String'));
    velMin = str2num(get(findobj('Tag', 'editMin'), 'String'));
    velTol = str2num(get(findobj('Tag', 'editTol'), 'String'))/100;
    applyVelocityFiltering(velMin, velMax, velTol);
    %Display
    subplot(3,1,1), cla, hold on, plot(dProcessed(:, TIMECOL),dProcessed(:, TORQUECOL),'ro' ), plot(d(:,TIMECOL), d(:,TORQUECOL)), hold off;
    xlabel('Time (mSec.)');    ylabel('Torque (N m)');
    drawnow;
    subplot(3,1,2), cla, hold on, plot(dProcessed(:, TIMECOL),dProcessed(:, ANGLECOL),'ro' ), plot(d(:,TIMECOL), d(:,ANGLECOL)), hold off;
    xlabel('Time (mSec.)');    ylabel('Angle (Degrees)');
    drawnow;
    subplot(3,1,3); cla; hold on; plot(dProcessed(:, TIMECOL),dProcessed(:, VELCOL),'ro' );plot(d(:,TIMECOL), d(:,VELCOL));    hold off;
    xlabel('Time (mSec.)');    ylabel('Velocity. (Deg/sec)');
    drawnow;
end
